///////////////////////////////////////////////////////////////////////////////////
//No part of this file can be copied or released without the consent of 
//Avalanche Technology
//										
///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////
//										
//	Avalanche Technology Inc., Proprietary and Confidential	   *
//										
// 	Release:  3.4    Date 01/26/2026  	
//							
//////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////
//  PART DESCRIPTION:
//
//  Technology: 40nm pMTJ STT-MRAM
//  Part:       AS3016B16/AS3032B16/AS3064B16
//
//  Description: 16/32/64 MEGABIT HIGH REL PARALLEL PERSISTENT SRAM MEMORY
//  Datasheet Revision: Rev U
//
////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////
//  FILE CONTENTS : MODEL OF PPSRAM 
//
////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////
// DENSITY SELECTION
//
// Uncomment one of the following lines to select the appropriate density. If
// no line is selected, the model will default to 16Mb
////////////////////////////////////////////////////////////////////////////////////
//`define DENSITY_16M
//`define DENSITY_32M
//`define DENSITY_64M

////////////////////////////////////////////////////////////////////////////////////
// MODULE DECLARATION                                                             //
////////////////////////////////////////////////////////////////////////////////////

`timescale 1ns/1ps

`ifdef DENSITY_16M
`define ADDR_WID    20 
`define ARRAY_BYTES 32'h100000
`elsif DENSITY_32M
`define ADDR_WID    21 
`define ARRAY_BYTES 32'h200000
`elsif DENSITY_64M
`define ADDR_WID    22 
`define ARRAY_BYTES 32'h400000
`else
`define ADDR_WID    20 
`define ARRAY_BYTES 32'h100000
`endif

`define NOCHK_tAVQV

module ppsram_as30xxb16(
  input                   En,
  input                   Wn,
  input [`ADDR_WID-1 : 0] ADDR,
  inout [15 : 0]          DQ,
  input                   Gn,
  input                   UBn,
  input                   LBn
  );

  /* Memory Array */
  reg [15:0] mem_array [0 : `ARRAY_BYTES-1];
  //reg mem_wpen;

  /* Timing Parameters */
  specify
    specparam tWLEH = 25;
    specparam tWLWH = 25;
    specparam tAVAV = 45;
    specparam tAVWL = 0;
    specparam tDVWH = 15;
    specparam tWHAX = 12;
    specparam tWHDX = 0;
    specparam tAVWH_GnH = 28;
    specparam tAVWH_GnL = 30;

    specparam tWLQZ = 15;
    specparam tWHQX = 3;

    specparam tAVQV = 45;
    specparam tELQV = 45;
    specparam tGLQV = 25;

    specparam tELQX = 3;
    specparam tGLQX = 0;
    specparam tAXQX = 3;

    specparam tEHQZ = 15;
    specparam tGHQZ = 15;

    specparam tBLQV = 25;
    specparam tBLQX = 0;
    //specparam tBHQZ = 10; //not modeled
  endspecify

  /* Output latch and enable */
  reg [15:0] dout;
  reg [15:0] dread; // internal data read
  reg dout_en;

  /* States */
  reg [7:0] state;
  localparam S_IDLE = 8'd0;
  localparam S_ENABLE = 8'd1;
  localparam S_WRWAIT = 8'd2;
  localparam S_WRDONE = 8'd3;
  localparam S_ERROR  = 8'hFF;

  /* En low counter */
  integer En_lctr;
  always@(negedge En) begin
    En_lctr <= 0;
    while (!En) begin
      #1 En_lctr <= En_lctr + 1;
    end
  end

  /* Wn counter */
  integer Wn_lctr;
  always@(negedge Wn) begin
    Wn_lctr <= 0;
    while (!Wn) begin
      #1 Wn_lctr <= Wn_lctr + 1;
    end
  end

  /* Gn low counter */
  integer Gn_lctr;
  always@(negedge Gn) begin
    Gn_lctr <= 0;
    while (!Gn) begin
      #1 Gn_lctr <= Gn_lctr + 1;
    end
  end

  /* UBn/LBn low counter */
  integer UBnLBn_lctr;
  always@(negedge UBn or negedge LBn) begin
    UBnLBn_lctr <= 0;
    while (!UBn || !LBn) begin
      #1 UBnLBn_lctr <= UBnLBn_lctr + 1;
    end
  end

  /* UBn low counter */
  integer UBn_lctr;
  always@(negedge UBn) begin
    UBn_lctr <= 0;
    while (!UBn) begin
      #1 UBn_lctr <= UBn_lctr + 1;
    end
    UBn_lctr <= 0;
  end

  /* LBn low counter */
  integer LBn_lctr;
  always@(negedge LBn) begin
    LBn_lctr <= 0;
    while (!LBn) begin
      #1 LBn_lctr <= LBn_lctr + 1;
    end
    LBn_lctr <= 0;
  end
  
  /* Gn high counter */
  integer Gn_hctr;
  always@(posedge Gn) begin
    Gn_hctr <= 0;
    while (Gn) begin
      #1 Gn_hctr <= Gn_hctr + 1;
    end
  end

  /* UBn/LBn high counter */
  integer UBnLBn_hctr;
  always@(posedge UBn or posedge LBn) begin
    UBnLBn_hctr <= 0;
    while (UBn || LBn) begin
      #1 UBnLBn_hctr <= UBnLBn_hctr + 1;
    end
  end

  /* En high counter */
  integer En_hctr;

  /* Address stable counter */
  integer ADDR_ctr;
  reg ADDR_ch;
  always@(ADDR) begin
    ADDR_ch <= 1'b1;
    #1 ADDR_ch <= 0;
  end
  always@(posedge ADDR_ch or negedge ADDR_ch) begin
    if(ADDR_ch)
      ADDR_ctr = ADDR_ctr + 1;
    else begin
      ADDR_ctr = 0;
      while(!ADDR_ch) begin
        #1;
        ADDR_ctr = ADDR_ctr + 1;
      end
    end
  end

  integer ADDR_ctr_q;
  always@(posedge ADDR_ch) begin
    ADDR_ctr_q = 0;
  end
  always@(negedge ADDR_ch) begin
    ADDR_ctr_q = ADDR_ctr_q + 1;
    #1;
    while(!ADDR_ch) begin
      ADDR_ctr_q = ADDR_ctr_q + 1;
      #1;
    end
  end

  /* Data stable counter */
  integer din_ctr;
  reg DQ_ch;
  always@(DQ) begin
    DQ_ch <= 1'b1;
    #1 DQ_ch <= 0;
  end
  always@(posedge DQ_ch or negedge DQ_ch) begin
    if(DQ_ch)
      din_ctr = din_ctr + 1;
    else begin
      din_ctr = 0;
      while(!DQ_ch) begin
        #1;
        din_ctr = din_ctr + 1;
      end
    end
  end

  /* Write delay counter */
  integer wrdly_ctr;
  reg wrdly_en;
  always@(state)
    if (state == S_WRDONE) begin
      wrdly_en <= 1;
    end
    else begin
      wrdly_en <= 0;
    end
  always begin
    if (wrdly_en) begin
      wrdly_ctr <= wrdly_ctr + 1;
      #1;
    end
    else begin
      wrdly_ctr <= 0;
      #1;
    end
  end

  /* Wn negedge counter */
  integer Wn_nectr;
  always@(negedge Wn)
    Wn_nectr = 0;
  always@(ADDR_ctr) 
    Wn_nectr = Wn_nectr + 1;

  /* State Machine */
  always@(posedge En
          or posedge Wn
          or negedge En 
          or negedge Wn
          //or negedge PGn
          or En_lctr
          or Wn_lctr
          or posedge ADDR_ch
          or wrdly_ctr
         )
    case(state)
      S_IDLE:
        if (!En) begin
          state <= S_ENABLE;
        end
        else begin
          state <= S_IDLE;
        end

      S_ENABLE:
        if (!Wn) begin
          if (ADDR_ctr >= tAVWL-1) begin
            state <= S_WRWAIT; // continue with write
          end
          else begin
            state <= S_ERROR; // ignore write
            $display("[MODEL] Write ignored. tAVWL violation");
          end
        end
        else if (En) begin
          state <= S_IDLE;
        end
        else
          state <= S_ENABLE;

      S_WRWAIT:
        if (Wn | En) begin
          if ((Wn_lctr >= tWLWH-1)
              && (En_lctr >= tWLEH-1)
              //&& (UBnLBn_lctr >= tWLEH-1)
             ) begin
            if (din_ctr >= tDVWH-2) begin
              if (Gn) begin
                if (ADDR_ctr >= tAVWH_GnH-2) begin
                  state <= S_WRDONE; // continue with write
                end
                else begin
                  state <= S_ERROR; // ignore write
                  $display("[MODEL] Write ignored. tAVWH violation");
                end
              end
              else begin
                if (ADDR_ctr >= tAVWH_GnL-2) begin
                  state <= S_WRDONE; // continue with write
                end
                else begin
                  state <= S_ERROR; // ignore write
                  $display("[MODEL] Write ignored. tAVWH violation");
                end
              end
            end
            else begin
              state <= S_ERROR; // ignore write
              $display("[MODEL] Write ignored. tDVWH violation");
            end
          end
          else begin
            state <= S_ERROR; // ignore write
            $display("[MODEL] Write ignored. tWLWH/tWLEH violation");
          end
        end
        else begin
          state <= S_WRWAIT;
        end

      S_WRDONE: begin
        if ((wrdly_ctr >= tWHAX)
            && (ADDR_ctr >= tAVAV-1)
           ) begin
          if (!En) begin
            state <= S_ENABLE;
          end
          else begin
            state <= S_IDLE;
          end
        end
        else if (ADDR_ch) begin 
          if (ADDR_ctr < tAVAV-1) begin
            $display("[MODEL] Write Ignored. tAVAV violation");
          end
          else if (wrdly_ctr < tWHAX) begin
            $display("[MODEL] Write Ignored. tWHAX violation");
          end
          state <= S_ERROR;
        end
        else begin
          state <= S_WRDONE;
        end
      end

      S_ERROR:
        if (En)
          state <= S_IDLE;
        else
          state <= S_ERROR;

      default:
        state <= S_ERROR;
    endcase

    /* Data in latch */
    reg [15:0] din;
    always@(posedge Wn or posedge En)
      din <= DQ;

    /* Write trigger */
    reg wr_trig;
    always@(wrdly_ctr or state)
      if ((state == S_WRDONE) && (wrdly_ctr >= tWHAX-1)) begin
        if (ADDR_ctr >= tAVAV-2) begin
          wr_trig <= 1'b1;
        end
        else begin
          wr_trig <= 0;
          //$display("[MODEL] Write ignored. tAVAV violation");
        end
      end
      else begin
        wr_trig <= 0;
      end

    reg UBn_q, LBn_q;
    always@(posedge wr_trig) begin
`ifndef DENSITY_64M
      case({UBn_q,LBn_q})
        2'b00: begin
          mem_array[ADDR] = din;
          $display("[MODEL] Data 0x%X written to Addr 0x%X", din, ADDR);
        end
        2'b01: begin
          mem_array[ADDR] = {din[15:8], mem_array[ADDR][7:0]};
          $display("[MODEL] Data 0x%X written to Addr 0x%X", 
                   {din[15:8], mem_array[ADDR][7:0]}, ADDR);
        end
        2'b10: begin
          mem_array[ADDR] = {mem_array[ADDR][15:8], din[7:0]};
          $display("[MODEL] Data 0x%X written to Addr 0x%X",
                   {mem_array[ADDR][15:8], din[7:0]}, ADDR);
        end
        2'b11: begin
          $display("[MODEL] Data not written to Addr 0x%X due to UBn and LBn both high", ADDR);
        end
      endcase
`else
          mem_array[ADDR] = din;
          $display("[MODEL] Data 0x%X written to Addr 0x%X", din, ADDR);
`endif
    end

    always@(posedge Wn or posedge En) begin
      if (UBn_lctr >= tWLEH-1)
        UBn_q <= 0;
      else
        UBn_q <= 1'b1;
    end

    always@(posedge Wn or posedge En) begin
      if (LBn_lctr >= tWLEH-1)
        LBn_q <= 0;
      else
        LBn_q <= 1'b1;
    end

    /* Internal read */
    always@(negedge Gn or negedge ADDR_ch or negedge En) begin
     if(!En && !Gn)
      dread <= mem_array[ADDR];
    end

    /* Output latch */
    always@(Gn_lctr)
      if ((En_lctr >= tELQV)
          || (Gn_lctr >= tGLQV)
          || (ADDR_ctr >= tAVQV-1)
`ifndef DENSITY_64M
          || (UBnLBn_lctr >= tBLQV)
`endif
         ) begin
        dout = dread;
      end

    /* Output enable state machine */
    localparam S_OEIDLE = 8'd0;
    localparam S_OEWAIT = 8'd1;
    localparam S_OEACTV = 8'd2;
    localparam S_OEDELY = 8'd3;
    reg [7:0] oestate;
    always@(En_lctr
            or En_hctr
            or Gn_lctr
            or Gn_hctr
           )begin
      case(oestate)
        S_OEIDLE: begin
          if (!En && !Gn && Wn && (state == S_IDLE || state == S_ENABLE)) begin
            if ((En_lctr >= tELQX-1)
                //&& (Gn_lctr >= tGLQX-1) // Comment out if tGLQX = 0
                //&& (UBn_LBn_lctr >= tBLQX-1) // Comment out if tBLQX = 0
               ) begin
              oestate <= S_OEACTV;
            end
            else begin
              oestate <= S_OEWAIT;
            end
          end
          else begin
            oestate <= S_OEIDLE;
          end
        end

        S_OEWAIT: begin
          if (En | Gn | !Wn) begin
            oestate <= S_OEIDLE;
          end
          else if ((En_lctr >= tELQX-1)
                   //&& (Gn_lctr >= tGLQX-1) // Comment out if tGLQX = 0
                   //&& (UBn_LBn_lctr >= tBLQX-1) // Comment out if tBLQX = 0
                  ) begin
            oestate <= S_OEACTV;
          end
          else begin
            oestate <= S_OEIDLE;
          end
        end

        S_OEACTV: begin
          if (!Wn) begin
            oestate <= S_OEIDLE;
          end
          else if (En | Gn) begin
            oestate <= S_OEDELY;
          end
          else begin
            oestate <= S_OEACTV;
          end
        end

        S_OEDELY: begin
          if ((En_hctr >= tEHQZ)
              && (Gn_hctr >= tGHQZ)
              || ADDR_ch
              //&& (UBnLBn_hctr >= tBHQZ)
             ) begin

            if (!En && !Gn) begin
              if ((En_lctr >= tELQX-1)
                  //&& (Gn_lctr >= tGLQX-1) // Comment out if tGLQX = 0;
                  //&& (UBn_LBn_lctr >= tBLQX-1) // Comment out if tBLQX = 0
                 ) begin
                oestate <= S_OEACTV;
              end
              else begin
                oestate <= S_OEWAIT;
              end
            end
            else begin
              oestate <= S_OEIDLE;
            end

          end
          else begin
            oestate <= S_OEDELY;
          end
        end

        default: begin
          oestate <= S_OEIDLE;
        end
      endcase
    end

    always@(Gn_hctr) begin
      if (oestate == S_OEDELY) begin
        En_hctr <= En_hctr + 1;
      end
      else begin
        En_hctr <= 0;
      end
    end


    /* Output enable */
    reg oeactv_hiz;
    always@(*)
      case(oestate)
        S_OEDELY: dout_en = 1'b1;
        S_OEACTV: dout_en = (oeactv_hiz)? 0 : 1'b1;
        default: dout_en = 0;
      endcase

    always@(*) begin
      if (oestate == S_OEACTV) begin
        /* 45ns access time */
        if ((En_lctr >= tELQV)
                 || (Gn_lctr >= tGLQV)
                 || (ADDR_ctr >= tAVQV-1)
`ifndef DENSITY_64M
                 || (UBnLBn_lctr >= tBLQV)
`endif
                ) begin
          oeactv_hiz <= 0;
        end
        /* Hiz after previous read */
        else if ((En_lctr >= tELQX)
            && (Gn_lctr >= tGLQX)
            && (ADDR_ctr >= tAXQX-1)
            ) begin
          oeactv_hiz <= 1'b1;
        end
        else begin
          oeactv_hiz <= 0;
        end
      end
      else begin
        oeactv_hiz <= 0;
      end
    end

`ifndef DENSITY_64M
  assign DQ[15:8] = (dout_en & !UBn)? dout[15:8] : 16'hz;
  assign DQ[7:0]  = (dout_en & !LBn)? dout[7:0]  : 16'hz;
`else
  assign DQ = (dout_en)? dout : 16'hz;
`endif

    /* Initialization */
    initial begin
      state = S_IDLE;
      dout = 0;
      En_lctr = 0;
      Gn_lctr = 0;
      Gn_hctr = 0;
      UBnLBn_lctr = 0;
      UBnLBn_hctr = 0;
    end

endmodule
